package service;

import java.util.Set;


public abstract class TransitionSystem<S,A> {
	
	public abstract Set<A> getActionsOfStateWithName(S from,String name) throws Exception;

	/**
	 * Return the name associated to the transition system
	 * @return the string contains the name of the transition system
	 */
	public abstract String getName();
	/**
	 * Set the name associated to the transition system
	 * @param name the new name of the transition system
	 * @throws Exception return a Exception if the string name is null
	 */
	public abstract void setName(String name) throws Exception;

	/**
	 * Add a new state to the transition system
	 * @param state the state that we want add to the transition system
	 * @throws Exception return a Exception if the argument state is null or if there is already a state with the same name in the transition system or if the state is initial and there is already a initial state in the transistion system
	 */
	public abstract void addState(S state) throws InvalidArgumentException, DuplicateStateException;
	/**
	 * Add a new action to the transition system
	 * @param stateFrom the source state of the action 
	 * @param stateTo the target state of the action
	 * @param a the action that 
	 * @throws Exception return a Exception if the argument state is null of if the stateFrom or stateTo don't exist in the transition system
	 */
	public abstract void addAction(S stateFrom,S stateTo,A a) throws Exception;
	/**
	 *	Returns a Set that contain all the states of the transition system 
	 * @return the Set described above
	 */
	public abstract Set<S> getAllStates();
	/**
	 *	Returns a Set that contain all the actions of the transition system 
	 * @return the Set described above
	 */
	public abstract Set<A> getAllActions();

	/**
	 *	Returns a Set that contain all the actions contained in the transistion system that connects the state "from" with the state "to" 
	 * @return the Set described above or a empty set
	 * @throws Exception return a Exception if the argument "from" or "to" are null or they aren't contained in the transition sytem
	 */
	public abstract Set<A> getActions(S from, S to)  throws Exception;

	/**
	 *	Returns a Set that contain all the actions contained in the transistion system that have the source state "from" 
	 * @return the Set described above or a empty set
	 * @throws Exception return a Exception if the argument "from" is null or they aren't contained in the transition sytem
	 */
	public abstract Set<A> getActionsOf(S from) throws Exception;
	/**
	 *	Returns if the transition system represented is valid or not 
	 * @return return true if the transition system is valid else return not
	 */
	public abstract boolean isValid();
	
	/**
	 * If the transition system is invalid return a string that contain the cause of non-validity
	 * else return null
	 */
	public abstract String getError();
	
	/**
	 * Return the target state of the action used as argument 
	 * @throws Exception This method throws a exception if the argument action is null or if the transition system doesn't contain the action
	 */
	public abstract S getTargetState(A action)  throws Exception;
	/**
	 * Return the state identified by the name passed as argument
	 * @param name the name of the state that we want search
	 * @throws Exception This method throws a exception if the argument name is null or if the transition system doesn't contain the action
	 */
	public abstract S getState(String name);
	/**
	 * Return the source state of the action used as argument 
	 * @throws Exception This method throws a exception if the argument action is null or if the transition system doesn't contain the action
	 */
	public abstract S getSourceState(A action) throws Exception;
	
	/** Return the initial state of the transition system
	 */
	public abstract S getInitial();
	public abstract boolean equals(Object o);
}